<?php
/**
 * Plugin Name: FG Magento to WooCommerce Premium Product Bundles module
 * Depends:		FG Magento to WooCommerce Premium
 * Plugin Uri:  https://www.fredericgilles.net/fg-magento-to-woocommerce/
 * Description: A plugin to migrate the Magento product bundle options as WooCommerce product bundles
 *				Needs the plugin «FG Magento to WooCommerce» to work
 *				Needs the WooCommerce Product Bundles plugin
 * Version:     1.1.3
 * Author:      Frédéric GILLES
 */

// Exit if accessed directly
if ( !defined( 'ABSPATH' ) ) exit;

add_action( 'admin_init', 'fgm2wc_product_bundles_test_requirements' );

if ( !function_exists( 'fgm2wc_product_bundles_test_requirements' ) ) {
	function fgm2wc_product_bundles_test_requirements() {
		new fgm2wc_product_bundles_requirements();
	}
}

if ( !class_exists('fgm2wc_product_bundles_requirements', false) ) {
	class fgm2wc_product_bundles_requirements {
		private $parent_plugin = 'fg-magento-to-woocommerce-premium/fg-magento-to-woocommerce-premium.php';
		private $required_premium_version = '2.66.0';

		public function __construct() {
			load_plugin_textdomain( 'fgm2wc_product_bundles', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
			if ( !is_plugin_active($this->parent_plugin) ) {
				add_action( 'admin_notices', array($this, 'product_bundles_error') );
			} else {
				$plugin_data = get_plugin_data(WP_PLUGIN_DIR . '/' . $this->parent_plugin);
				if ( !$plugin_data or version_compare($plugin_data['Version'], $this->required_premium_version, '<') ) {
					add_action( 'admin_notices', array($this, 'product_bundles_version_error') );
				}
			}
			if ( !is_plugin_active('woocommerce-product-bundles/woocommerce-product-bundles.php') ) {
				add_action( 'admin_notices', array($this, 'product_bundles_required') );
			}
			
		}
		
		/**
		 * Print an error message if the Premium plugin is not activated
		 */
		public function product_bundles_error() {
			echo '<div class="error"><p>[fgm2wc_product_bundles] ' . __('The Product Bundles module needs the «FG Magento to WooCommerce Premium» plugin to work. Please install and activate <strong>FG Magento to WooCommerce Premium</strong>.', 'fgm2wc_product_bundles') . '<br /><a href="https://www.fredericgilles.net/fg-magento-to-woocommerce/" target="_blank">https://www.fredericgilles.net/fg-magento-to-woocommerce/</a></p></div>';
		}
		
		/**
		 * Print an error message if the Premium plugin is not at the required version
		 */
		public function product_bundles_version_error() {
			printf('<div class="error"><p>[fgm2wc_product_bundles] ' . __('The Product Bundles module needs at least the <strong>version %s</strong> of the «FG Magento to WooCommerce Premium» plugin to work. Please install and activate <strong>FG Magento to WooCommerce Premium</strong> at least the <strong>version %s</strong>.', 'fgm2wc_product_bundles') . '<br /><a href="https://www.fredericgilles.net/fg-magento-to-woocommerce/" target="_blank">https://www.fredericgilles.net/fg-magento-to-woocommerce/</a></p></div>', $this->required_premium_version, $this->required_premium_version);
		}
		
		/**
		 * Print an error message if the WooCommerce Product Bundles plugin is not activated
		 */
		public function product_bundles_required() {
			printf('<div class="error"><p>[fgm2wc_product_bundles] ' . __('<a href="%s" target="_blank">WooCommerce Product Bundles</a> is required. Please install and activate it.', 'fgm2wc_product_bundles') . '</p></div>', 'https://woocommerce.com/products/product-bundles/?aff=3777');
		}
	}
}

if ( !defined('WP_LOAD_IMPORTERS') && !defined('DOING_AJAX') ) return;

add_action( 'plugins_loaded', 'fgm2wc_product_bundles_load', 25 );

if ( !function_exists( 'fgm2wc_product_bundles_load' ) ) {
	function fgm2wc_product_bundles_load() {
		if ( !defined('FGM2WCP_LOADED') ) return;

		load_plugin_textdomain( 'fgm2wc_product_bundles', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
		
		global $fgm2wcp;
		new fgm2wc_product_bundles($fgm2wcp);
	}
}

if ( !class_exists('fgm2wc_product_bundles', false) ) {
	class fgm2wc_product_bundles {
		
		private $plugin;
		private $imported_product_bundles_count = 0;
		
		/**
		 * Sets up the plugin
		 *
		 */
		public function __construct($plugin) {
			
			$this->plugin = $plugin;
			
			add_filter('fgm2wc_pre_display_admin_page', array($this, 'process_admin_page'), 11, 1);
			add_action('fgm2wc_post_empty_database', array($this, 'delete_product_bundles_data'));
			add_action('fgm2wc_post_import_products', array($this, 'import_product_bundle_subproducts'));
			add_action('fgm2wc_post_import_products', array($this, 'import_product_bundles'));
		}

		/**
		 * Add information to the admin page
		 * 
		 * @param array $data
		 * @return array
		 */
		public function process_admin_page($data) {
			$data['title'] .= ' ' . __('+ Product Bundles module', __CLASS__);
			$data['description'] .= "<br />" . sprintf(__('The Product Bundles module will also import the Magento product bundle options as WooCommerce product bundles. It requires the <a href="%s" target="_blank">WooCommerce Product Bundles</a> plugin.', __CLASS__), 'https://woocommerce.com/products/product-bundles/?aff=3777');
			return $data;
		}
		
		/**
		 * Delete all woocommerce data
		 *
		 */
		public function delete_product_bundles_data() {
			global $wpdb;
			
			$wpdb->show_errors();
			
			$sql_queries = array();
			$sql_queries[] = <<<SQL
-- Disable foreign key checks
SET FOREIGN_KEY_CHECKS=0;
SQL;

			$sql_queries[] = <<<SQL
-- Delete WooCommerce bundled item meta
TRUNCATE {$wpdb->prefix}woocommerce_bundled_itemmeta
SQL;

			$sql_queries[] = <<<SQL
-- Delete WooCommerce bundled items
TRUNCATE {$wpdb->prefix}woocommerce_bundled_items
SQL;

			$sql_queries[] = <<<SQL
-- Enable foreign key checks
SET FOREIGN_KEY_CHECKS=1;
SQL;

			// Execute SQL queries
			if ( count($sql_queries) > 0 ) {
				foreach ( $sql_queries as $sql ) {
					$wpdb->query($sql);
				}
			}
			$wpdb->hide_errors();
		}
		
		/**
		 * Import the products used by Product Bundles
		 * 
		 */
		public function import_product_bundle_subproducts() {
			$this->plugin->log(__('Importing Product Bundles subproducts...', $this->plugin->get_plugin_name()));
			
			$products = $this->get_all_product_bundle_subproducts();
			foreach ( $products as $product ) {
				if ( !in_array($product['entity_id'], array_keys($this->plugin->imported_products[$this->plugin->default_language])) ) {
					$new_post_id = $this->plugin->import_product($product, $this->plugin->default_language);
					
					// Hook for doing other actions after importing the product
					do_action('fgm2wc_post_import_product', $new_post_id, $product);
				}
			}
		}
		
		/**
		 * Get all the Magento sub products
		 * 
		 * @return array of products
		 */
		private function get_all_product_bundle_subproducts() {
			$products = array();
			$prefix = $this->plugin->plugin_options['prefix'];
			
			$sql = "
				SELECT DISTINCT p.entity_id, p.type_id, p.sku, p.created_at
				FROM ${prefix}catalog_product_entity p
				INNER JOIN ${prefix}catalog_product_bundle_selection bs ON bs.product_id = p.entity_id
				ORDER BY p.entity_id
			";
			$products = $this->plugin->magento_query($sql);
			
			return $products;
		}
		
		/**
		 * Import the Magento product bundle options as Product Bundles
		 *
		 */
		public function import_product_bundles() {
			if ( !class_exists('WC_PB_DB', false) ) {
				$this->plugin->display_admin_error(sprintf(__('<a href="%s" target="_blank">WooCommerce Product Bundles</a> is required. Please install and activate it.', 'fgm2wc_product_bundles'), 'https://woocommerce.com/products/product-bundles/?aff=3777'));
				return;
			}
			$this->plugin->log(__('Importing Product Bundles...', $this->plugin->get_plugin_name()));
			$this->imported_product_bundles_count = 0;
			
			$products = $this->get_products_with_bundle_options();
			foreach ( $products as $product ) {
				if ( isset($this->plugin->imported_products[$this->plugin->default_language][$product['entity_id']]) ) {
					$new_product_id = $this->plugin->imported_products[$this->plugin->default_language][$product['entity_id']];
					$subproducts = $this->get_product_bundle_subproducts($product['entity_id']);
					if ( !empty($subproducts) ) {
						$this->import_bundled_items($new_product_id, $subproducts, $this->plugin->default_language);
						do_action('fgm2wc_post_import_bundled_items', $new_product_id, $subproducts, $this);
					}
				}
			}
			
			$this->plugin->display_admin_notice(sprintf(_n('%d Product Bundle imported', '%d Product Bundles imported', $this->imported_product_bundles_count, $this->plugin->get_plugin_name()), $this->imported_product_bundles_count));
		}
		
		/**
		 * Get the Magento products containing bundle options
		 * 
		 * @return array of products
		 */
		private function get_products_with_bundle_options() {
			$options = array();
			$prefix = $this->plugin->plugin_options['prefix'];

			$sql = "
				SELECT DISTINCT bs.parent_product_id AS entity_id
				FROM ${prefix}catalog_product_bundle_selection bs
			";
			$options = $this->plugin->magento_query($sql);
			
			return $options;
		}
		
		/**
		 * Get the Magento sub products
		 * 
		 * @param int $parent_product_id Parent product ID
		 * @return array of products
		 */
		private function get_product_bundle_subproducts($parent_product_id) {
			$products = array();
			$prefix = $this->plugin->plugin_options['prefix'];
			
			$sql = "
				SELECT DISTINCT bs.option_id, bs.product_id, bs.position, bs.selection_price_value, bs.selection_qty, bs.selection_can_change_qty
				FROM ${prefix}catalog_product_bundle_selection bs
				WHERE bs.parent_product_id = '$parent_product_id'
				ORDER BY bs.position, bs.option_id
			";
			$products = $this->plugin->magento_query($sql);
			
			return $products;
		}
		
		/**
		 * Import the bundled items for a product
		 * 
		 * @since 1.1.0
		 * 
		 * @param int $new_product_id WP product ID
		 * @param array $subproducts Sub products
		 * @param int $language Store ID
		 */
		public function import_bundled_items($new_product_id, $subproducts, $language) {
			$price = get_post_meta($new_product_id, '_price', true);
			$subproducts_found = false;
			foreach ( $subproducts as $subproduct ) {
				if ( isset($this->plugin->imported_products[$language][$subproduct['product_id']]) ) {
					$subproducts_found = true;
					$subproduct_wp_id = $this->plugin->imported_products[$language][$subproduct['product_id']];
					if ( !$this->bundled_item_exists($new_product_id, $subproduct_wp_id) ) {
						if ( $this->import_bundled_item($new_product_id, $subproduct_wp_id, $subproduct, $price) ) {
							$this->imported_product_bundles_count++;
						}
					}
				}
			}
			if ( $subproducts_found ) {
				// Set the product type as "bundle"
				wp_set_object_terms($new_product_id, $this->plugin->product_types['bundle'], 'product_type', false);
				
				// Set the bundle prices
				$regular_price = get_post_meta($new_product_id, '_regular_price', true);
				$sale_price = get_post_meta($new_product_id, '_sale_price', true);
				update_post_meta($new_product_id, '_wc_pb_base_price', $price);
				update_post_meta($new_product_id, '_wc_pb_base_regular_price', $regular_price);
				update_post_meta($new_product_id, '_wc_pb_base_sale_price', $sale_price);
			}
		}
		
		/**
		 * Check if the bundled item already exists
		 * 
		 * @since 1.1.2
		 * 
		 * @global object $wpdb
		 * @param int $parent_product_id Parent product ID
		 * @param int $subproduct_id Child product ID
		 * @return boolean Bundle item exists?
		 */
		private function bundled_item_exists($parent_product_id, $subproduct_id) {
			global $wpdb;
			$exists = false;
			
			$table_name = $wpdb->prefix . 'woocommerce_bundled_items';
			$sql = $wpdb->prepare("
					SELECT COUNT(*) FROM $table_name
					WHERE product_id = %s
					AND bundle_id = %s
			", $subproduct_id, $parent_product_id);
			$result = $wpdb->get_var($sql);
			$exists = $result > 0;
			return $exists;
		}
		
		/**
		 * Import a bundled item
		 * 
		 * @param int $parent_product_id Parent product ID
		 * @param int $subproduct_id Subproduct ID
		 * @param array $subproduct Subproduct data
		 * @param float $price Main product price
		 * @return false|int Result
		 */
		private function import_bundled_item($parent_product_id, $subproduct_id, $subproduct, $price) {
			if ( $subproduct['selection_can_change_qty'] == 0 ) {
				// The customer can not choose the quantity
				$quantity_max = $subproduct['selection_qty'];
				$optional = 'no';
			} else {
				// The customer can choose the quantity
				$quantity_max = '';
				$optional = 'yes';
			}
			$priced_individually = (floatval($subproduct['selection_price_value']) == 0.0) && ($price != 0.0)? 'no' : 'yes';
			
			return WC_PB_DB::add_bundled_item(array(
				'bundle_id'  => $parent_product_id,
				'product_id' => $subproduct_id,
				'meta_data'  => array(
					'quantity_min'			=> $subproduct['selection_qty'],
					'quantity_max'			=> $quantity_max,
					'priced_individually'	=> $priced_individually,
					'optional'				=> $optional,
				),
			));
		}
		
	}
}
